/* batch_RunScored_R123_precomp.jsx — RunScored with Precomp instances
   - Finds all layers named AE_RS_PRECOMP_NAME on root comp (RunScored-192).
   - For each distinct source comp referenced, replace its inner RunScoredText image(s)
     with RunScoredText-{RUNNUM}.png (reuse footage if already imported).
   - Set all Tint effects inside that precomp to Smart Secondary (Amount=100).
   - Replace LogoPreComp ▸ Logo (and optional flat logo layer).
   - Render per team, per run (R-1..R-4).
*/

(function () {
  // --- load common helpers (force reload) ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) { try { delete $.global.GL; } catch(e) { $.global.GL = undefined; } $.evalFile(File(COMMON)); }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();
  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT     = GL.env("AE_PROJECT", null);
  var CSV_PATH    = GL.env("AE_CSV", null);
  var COMP_NAME   = GL.env("AE_COMP","RunScored-192");

  var RS_PRECOMP_NAME = GL.cleanValue(GL.env("AE_RS_PRECOMP_NAME","RunScoredText-PreComp"));

  var RSTEXT_DIR   = GL.env("AE_RSTEXT_DIR","");
  var RS_BASENAME  = GL.env("AE_RS_BASENAME","RunScoredText");
  var RS_INNER_PREF= GL.env("AE_RS_INNER_PREFIX","RunScoredText"); // empty -> all footage layers
  var RUNNUM       = parseInt(GL.env("AE_RUNNUM","1"),10) || 1;

  var LAYER_LOGO_PRECOMP       = GL.cleanValue(GL.env("AE_LAYER_LOGO_PRECOMP","LogoPreComp"));
  var SUBLAYER_LOGO_IN_PRECOMP = GL.cleanValue(GL.env("AE_SUBLAYER_LOGO_IN_PRECOMP","Logo"));
  var LAYER_LOGO2              = GL.cleanValue(GL.env("AE_LAYER_LOGO2","TeamLogo")); // optional

  var OUTDIR   = GL.env("AE_OUTDIR","");
  var PATH_TPL = GL.env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME= GL.env("AE_ANIM","R-1");
  var RS_TPL   = GL.env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL   = GL.env("AE_OM_TEMPLATE","PNG Sequence");

  var LEAGUE   = GL.env("AE_LEAGUE","");
  var PURGE    = (GL.env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER= (GL.env("AE_NO_RENDER","0")==="1");
  var QUIT_APP = (GL.env("AE_QUIT","1")==="1");

  var logoOpts = { dir:GL.env("AE_LOGO_DIR",""),
                   tpl:GL.env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}"),
                   exts:GL.env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd") };

  // --- open project & csv ---
  if(!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT); if(!aep.exists) GL.fail("AE_PROJECT not found: "+PROJECT);
  try{
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  }catch(e){ app.open(aep); }

  if(!CSV_PATH) GL.fail("AE_CSV env not set.");
  if(!LEAGUE || GL.cleanValue(LEAGUE)==="") GL.fail("AE_LEAGUE is required.");

  var rows   = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams  = GL.buildTeams(rows, LEAGUE);
  var todo   = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if(!todo.length) GL.fail("No teams matched league: "+LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if(!comp) GL.fail("Comp not found: "+COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("RunScored (PreComp) Batch");

  // --- helpers ---
  function _safe01(c){ return GL.safeColor(GL.ensureNonBlack(c)); }
  function _normLoose(s){ return String(GL.cleanValue(s||"")).toLowerCase().replace(/[^a-z0-9]/g,""); }

  function findOrImportFootage(filePath){
    var f = new File(filePath);
    if (!f.exists) { $.writeln("File not found: "+filePath); return null; }
    var fs = f.fsName;
    for (var i=1;i<=app.project.numItems;i++){
      var it = app.project.item(i);
      try{ if (it instanceof FootageItem && it.file && it.file.fsName === fs) return it; }catch(e){}
    }
    try{
      var io = new ImportOptions(f);
      if (!io.canImportAs(ImportAsType.FOOTAGE)) { $.writeln("Cannot import as FOOTAGE: "+filePath); return null; }
      io.importAs = ImportAsType.FOOTAGE;
      return app.project.importFile(io);
    }catch(e){
      $.writeln("Import failed: "+filePath+"  ("+e+")");
      return null;
    }
  }

  // collect distinct source comps referenced by layers named RS_PRECOMP_NAME
  function collectRSPrecompSources(root){
    var seen = {}; var out = [];
    var want = RS_PRECOMP_NAME;
    for (var i=1;i<=root.numLayers;i++){
      var L = root.layer(i);
      if (L.name !== want) continue;
      if (L.source && (L.source instanceof CompItem)){
        var key = L.source.id; // unique for the session
        if (!seen[key]){ seen[key] = true; out.push(L.source); }
      }
    }
    return out;
  }

  // replace all (or name-matching) footage layers inside a comp with given FootageItem
  function replaceFootageInsideComp(targetComp, footage, namePrefix){
    var hits = 0;
    var pref = String(namePrefix||"");
    var useFilter = pref.length > 0;
    for (var i=1;i<=targetComp.numLayers;i++){
      var L = targetComp.layer(i);
      try{
        var isFootage = (L.source && (L.source instanceof FootageItem));
        if (!isFootage) continue;
        if (useFilter){
          if (L.name.indexOf(pref) !== 0) continue; // must start with prefix
        }
        L.replaceSource(footage, false);
        hits++;
      }catch(e){}
    }
    return hits;
  }

  // set all Tint effects inside a comp (deep) to color01 (Amount=100)
  function setAllTintsDeep(compItem, color01, depth){
    depth = depth||0; if (!compItem || depth>12) return 0;
    var hits=0, c = _safe01(color01);
    for (var i=1;i<=compItem.numLayers;i++){
      var L = compItem.layer(i);

      var fx = L.property("Effects");
      if (fx){
        for (var j=1;j<=fx.numProperties;j++){
          var eff = fx.property(j);
          try{
            if (eff && eff.matchName === "ADBE Tint"){
              for (var k=1;k<=eff.numProperties;k++){
                var p = eff.property(k);
                if (!p) continue;
                if (p.propertyValueType === PropertyValueType.COLOR) p.setValue(c);
                if (p.propertyValueType === PropertyValueType.ONE_D && /Amount/i.test(p.name)) p.setValue(100);
              }
              hits++;
            }
          }catch(e){}
        }
      }

      if (L.source && (L.source instanceof CompItem)){
        hits += setAllTintsDeep(L.source, c, depth+1);
      }
    }
    return hits;
  }

  // One-time per run: resolve per-run image & footage
  var rsPath = (function(){
    var dir = RSTEXT_DIR.replace(/\/+$/,"");
    var fp  = dir + "/" + RS_BASENAME + "-" + RUNNUM + ".png";
    var f = new File(fp);
    if (!f.exists) $.writeln("RunScoredText missing: "+fp);
    return f.exists ? f.fsName : null;
  })();
  var rsFootage = rsPath ? findOrImportFootage(rsPath) : null;

  // --- team loop ---
  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    var smart = GL.computeSmartColors( GL.safeColor(t.primary), GL.safeColor(t.secondary) );
    var P = smart.primary, S = smart.secondary; // we only need S here for tinting

    // Update all distinct RunScoredText-PreComp sources
    var subComps = collectRSPrecompSources(comp);
    for (var s=0; s<subComps.length; s++){
      var sc = subComps[s];
      if (rsFootage){
        replaceFootageInsideComp(sc, rsFootage, RS_INNER_PREF); // prefix filter or all footage
      }
      setAllTintsDeep(sc, S, 0); // tint inside precomp to Smart Secondary
    }

    // Logos
    var logoPath = (function(league,abbr,opts){
      var dir = (opts.dir||"").replace(/\/+$/,"");
      var tpl = (opts.tpl||"{league}/{abbr}");
      var exts = String(opts.exts||"png").split(",");
      var rel = tpl.replace("{league}", String(league||""))
                   .replace("{abbr}",   String(abbr||""));
      for (var i=0;i<exts.length;i++){
        var fp = dir + "/" + rel + "." + GL.cleanValue(exts[i]);
        var f = new File(fp);
        if (f.exists) return f.fsName;
      }
      return null;
    })(t.league, t.abbr, logoOpts);

    if (logoPath){
      (function replaceLogoInsidePrecompViaPath(rootComp, precompLayerName, innerLayerName, logoFilePath){
        var preL = GL.getLayer(rootComp, precompLayerName);
        if (!preL || !preL.source || !(preL.source instanceof CompItem)) return;
        var subComp = preL.source;
        var footage = findOrImportFootage(logoFilePath);
        if (!footage) return;
        for (var i=1;i<=subComp.numLayers;i++){
          var L = subComp.layer(i);
          if (L.name === innerLayerName && L.source){
            try{ L.replaceSource(footage, false); }catch(e){}
          }
        }
      })(comp, LAYER_LOGO_PRECOMP, SUBLAYER_LOGO_IN_PRECOMP, logoPath);

      if (LAYER_LOGO2){
        var flat = GL.getLayer(comp, LAYER_LOGO2);
        if (flat){
          var foot = findOrImportFootage(logoPath);
          if (foot && flat.source){ try{ flat.replaceSource(foot, false); }catch(e){} }
        }
      }
    }

    // Snap for crisp PNGs
    GL.snapCompTextForPixelArt(comp);
    if (PURGE && app.purge){ try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

    // Render
    if (!NO_RENDER){
      var lc    = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
  if (QUIT_APP) app.quit();
})();
